;;;
;;; nibimage.lisp
;;;
;;; read .NIB disk images used by Apple II emulators.
;;;
;;; Joseph A. Oswald, III
;;;
;;; 28 January 2000
;;;
;;; $Id$
;;;
;;; $Log$
;;;

(defconstant +standard-nib-track-length+ 6656 
  "Number of bytes per track in a standard .NIB nibble disk image.")

(defconstant +standard-nib-track-count+ 35)

(defconstant +standard-nib-length+ (* +standard-nib-track-length+ 
                                      +standard-nib-track-count+))

(defclass nib-image ()
  ((track-vector :accessor track-vector :initarg :track-vector)))

(defun read-nib-image-stream (stream)
  (let ((track-vector (make-array +standard-nib-track-count+)))
    (dotimes (track +standard-nib-track-count+)
      (let ((track-data (make-array +standard-nib-track-length+ 
                                    :element-type '(unsigned-byte 8))))
        (dotimes (i +standard-nib-track-length+)
          (setf (aref track-data i) (read-byte stream)))
        (setf (aref track-vector track) track-data)))
    
    (make-instance 'nib-image :track-vector track-vector)))

(defun read-nib-image (pathname)
  (with-open-file (str pathname :element-type '(unsigned-byte 8)
                       :direction :input)
    (read-nib-image-stream str)))

(defmethod write-nib-image ((image nib-image) stream)
  (let* ((tv (track-vector image)))
    (dotimes (trk (length tv))
      (let ((trk-data (aref tv trk)))
        (dotimes (i (length trk-data))
          (write-byte (aref trk-data i) stream))))))

(defmethod track ((image nib-image) track)
  (when (or (< track 0) (>= track +standard-nib-track-count+))
    (error "Illegal track ~D." track))

  (aref (track-vector image) track))


(defun four-and-four-decode (buffer offset)
  "Decode numbers in the four-and-four decoding system. This encodes 
a single eight-bit byte into two eight-bit bytes, presumed to be 
located in BUFFER at array elements OFFSET and OFFSET+1."
  (let ((first-byte (aref buffer offset))
        (second-byte (aref buffer (1+ offset))))
    (+ (* 2 (logxor #xAA first-byte)) (logxor #xAA second-byte))))


(defconstant +dos33-write-translate+ 
#(#x96 #x97 #x9A #x9B #x9D #x9E #x9F #xA6
  #xA7 #xAB #xAC #xAD #xAE #xAF #xB2 #xB3
  #xB4 #xB5 #xB6 #xB7 #xB9 #xBA #xBB #xBC
  #xBD #xBE #xBF #xCB #xCD #xCE #xCF #xD3
  #xD6 #xD7 #xD9 #xDA #xDB #xDC #xDD #xDE
  #xDF #xE5 #xE6 #xE7 #xE9 #xEA #xEB #xEC
  #xED #xEE #xEF #xF2 #xF3 #xF4 #xF5 #xF6
  #xF7 #xF9 #xFA #xFB #xFC #xFD #xFE #xFF) 
"Write translate codes in DOS 3.3.
Compare $BA29,X for X=0..$3F.")

(defconstant +dos33-read-translate+ 
#(#xB3 #xC5 #xAA #xA0 #x82 #xC5 #xB3 #xB3
  #xAA #x88 #x82 #xC5 #xB3 #xB3 #xAA #x88
  #x82 #xC5 #xC4 #xB3 #xB0 #x88 #x00 #x01
  #x98 #x99 #x02 #x03 #x9C #x04 #x05 #x06
  #xA0 #xA1 #xA2 #xA3 #xA4 #xA5 #x07 #x08
  #xA8 #xA9 #xAA #x09 #x0A #x0B #x0C #x0D
  #xB0 #xB1 #x0E #x0F #x10 #x11 #x12 #x13
  #xB8 #x14 #x15 #x16 #x17 #x18 #x19 #x1A
  #xC0 #xC1 #xC2 #xC3 #xC4 #xC5 #xC6 #xC7
  #xC8 #xC9 #xCA #x1B #xCC #x1C #x1D #x1E
  #xD0 #xD1 #xD2 #x1F #xD4 #xD5 #x20 #x21
  #xD8 #x22 #x23 #x24 #x25 #x26 #x27 #x28
  #xE0 #xE1 #xE2 #xE3 #xE4 #x29 #x2A #x2B
  #xE8 #x2C #x2D #x2E #x2F #x30 #x31 #x32
  #xF0 #xF1 #x33 #x34 #x35 #x36 #x37 #x38
  #xF8 #x39 #x3A #x3B #x3C #x3D #x3E #x3F) 
"Nibble-to-data conversion table. Compare $BA00,Y in DOS 3.3, where Y >= #x80"
)

(defconstant +dos33-sector-translate+
#(#x0 #xD #xB #x9
  #x7 #x5 #x3 #x1
  #xE #xC #xA #x8
  #x6 #x4 #x2 #xF) 
"compare to DOS 3.3 $BFB8..$BFC7 and in boot sector.")

(defconstant +dos33-address-prolog+
  #(#xd5 #xaa #x96) "Bytes indicating the beginning of an address field.")

(defconstant +dos33-address-epilog+
  #(#xde #xaa #xeb) "Bytes indicating the end of an address field.")

(defconstant +dos33-data-prolog+
#(#xd5 #xaa #xad) "Bytes indicating the beginning of a data field.")

(defconstant +dos33-data-epilog+
  +dos33-address-epilog+ "Bytes indicating the end of a data field.")

(defun xor-read-bytes (buffer offset length
                              &key (read-translate-table
                                    +dos33-read-translate+)
                              (initial-accumulator 0))
  "Read in disk bytes, translating using READ-TRANSLATE-TABLE, continuously 
XOR'ing to create a buffer of translated bytes that can be denibblized.
INITIAL-ACCUMULATOR should specify a seed for the XOR process.

Returns two values: an array of translated data, and the last translated
byte, which can be used to verify a data checksum.

Check out $B8F4 in DOS 3.3.
This function will return an array containing the values in 

$BC55,$BC54,.....$BC00,$BB00,$BB01,....,$BBFF before de-nibblizing."

  (let ((acc initial-accumulator)
        (destbuf (make-array length 
                             :element-type '(unsigned-byte 8)
                             :initial-element 0)))
    (labels ((lookup-byte (b)
               (when (< b #x80)
                 (error "Illegal disk nibble #x~X." b))
               (aref read-translate-table (- b #x80))))
      (dotimes (i length (values destbuf acc))
        (setf acc (logxor acc (lookup-byte (circular-aref buffer (+ offset i)))))
        (setf (aref destbuf i) acc)))))

(defun swap-2-bit-number (n)
  (ecase n
    (#b00 #b00)
    (#b01 #b10)
    (#b10 #b01)
    (#b11 #b11)))
    
(defun denibblize-dos33-read (buffer)
  
  (unless (= (length buffer) 342)
    (error "Incorrect buffer length."))
  
  (let ((destbuf (make-array 256 :element-type '(unsigned-byte 8)
                             :initial-element 0)))
    (dotimes (i 256 destbuf)
      (multiple-value-bind (low-order-2-shift low-order-2-location) 
                           (floor i #x56)
        (let ((low-order-2-bits-swapped (ldb (byte 2 (* 2 low-order-2-shift))
                                             (aref buffer 
                                                   low-order-2-location)))
              (high-order-6-bits (aref buffer (+ i #x56))))
          (setf (ldb (byte 6 2) (aref destbuf i)) high-order-6-bits
                (ldb (byte 2 0) (aref destbuf i)) (swap-2-bit-number
                                                   low-order-2-bits-swapped)))))))




(defmethod read-sector-data ((image nib-image) track offset 
                             &key (data-prolog +dos33-data-prolog+)
                             (data-epilog +dos33-data-epilog+)
                             (initial-accumulator 0)
                             (read-translate-table +dos33-read-translate+))
  (let* ((data (aref (track-vector image) track))
         (found-header (circular-verify-subseq data-prolog data offset #'=)))
    (if found-header
      (multiple-value-bind (nibblized-data checksum)
                           (xor-read-bytes data 
                                           (+ offset (length data-prolog))
                                           342 :initial-accumulator initial-accumulator
                                           :read-translate-table read-translate-table)
        (let ((read-checksum (aref read-translate-table
                                   (- (circular-aref data 
                                                     (+ offset (length data-prolog) 342))
                                      128)))
              (found-epilog (circular-verify-subseq data-epilog data 
                                                         (+ offset (length data-prolog) 343) #'=)))
          (values (denibblize-dos33-read nibblized-data)
                  (if (= checksum read-checksum)
                    t
                    (cons checksum read-checksum))
                  found-epilog)))
      nil)))
                  
    

(defun parse-address-sequence (seq offset &optional (address-prolog +dos33-address-prolog+)
                                   (address-epilog +dos33-address-epilog+))
  (let ((start (circular-position-subseq address-prolog seq :start offset :test #'=)))
    (unless start
      (error "Couldn't find address header."))
    (let ((address-block (circular-subseq seq (+ start 3) (+ start 11 (length address-epilog)))))
      (let ((volume (four-and-four-decode address-block 0))
            (track (four-and-four-decode address-block 2))
            (physical-sector (four-and-four-decode address-block 4))
            (checksum (four-and-four-decode address-block 6))
            (epilog (verify-subsequence address-epilog address-block 8 #'=)))
        (values start volume track physical-sector 
                checksum (if epilog
                           epilog
                           (subseq address-block 8 
                                   (+ 8 (length address-epilog)))))))))
                    
         
(defmethod parse-address ((image nib-image) track offset &optional 
                          (address-prolog +dos33-address-prolog+)
                          (address-epilog +dos33-address-epilog+))
  (parse-address-sequence (track image track) offset address-prolog address-epilog))

(defmethod track-sector-address ((image nib-image) track physical-sector 
                                 &optional (address-prolog +dos33-address-prolog+)
                                 (address-epilog +dos33-address-epilog+))
  (let ((trk (track image track)))
    (do ((offset 0)
         (found nil))
        ((or found (not offset)) (if offset 
                                   (values offset
                                           (circular-verify-subseq address-epilog trk
                                                                   (+ offset 
                                                                      (length address-prolog) 
                                                                      8) 
                                                                   #'=))
                                   offset))

      (multiple-value-bind (found-start vol log-trk found-sector checksum found-epilog)
                           (parse-address-sequence trk offset address-prolog address-epilog)
        (if found-start
          (if (= found-sector physical-sector)
            (setf found t offset found-start)
            (setf offset (1+ found-start)))
          (setf offset nil))))))

(defmethod track-sector ((image nib-image) track logical-sector
                         &key (sector-translate +dos33-sector-translate+)
                         (address-prolog +dos33-address-prolog+)
                         (address-epilog +dos33-address-epilog+)
                         (data-prolog +dos33-data-prolog+)
                         (data-epilog +dos33-data-epilog+)
                         (read-translate-table +dos33-read-translate+))

  (unless (and (>= logical-sector 0)
               (< logical-sector (length sector-translate)))
    (error "Illegal logical sector ~D." logical-sector))

  (let* ((physical-sector (aref sector-translate logical-sector))
         (offset (track-sector-address image track physical-sector 
                                       address-prolog address-epilog)))
    (unless offset
      (error "Could not locate track ~D physical sector ~D." track physical-sector))

    (let ((data-start-offset (circular-position-subseq data-prolog (track image track)
                                                       :start offset)))
      (unless data-start-offset
        (error "Could not locate start of data section."))

      (read-sector-data image track data-start-offset :data-prolog data-prolog
                        :data-epilog data-epilog
                        :read-translate-table read-translate-table))))

            
          
